﻿using System;
using System.Linq;
using System.ServiceModel;
using System.Collections.Generic;

namespace ChatServerLibrary
{
    [ServiceBehavior(
        InstanceContextMode = InstanceContextMode.Single,
        ConcurrencyMode = ConcurrencyMode.Reentrant)]
    public class ChatService : IChatService
    {
        private Dictionary<IChatClient, string> clientsAndNames =
            new Dictionary<IChatClient, string>();

        public bool Connect(string name)
        {
            if (clientsAndNames.ContainsValue(name))
            {
                // Imię jest już używane, dlatego połączenie zostaje odrzucone.
                return false;
            }

            IChatClient clientCallback =
                OperationContext.Current.GetCallbackChannel<IChatClient>();

            // clientsAndNames to stan współdzielony, ale nie następuje tu blokada,
            // ponieważ zakładamy, że tryb ConcurrentMode.Reentrant zapewnia
            // podawanie tylko jednego komunikatu naraz.
            clientsAndNames.Add(clientCallback, name);
            Console.WriteLine(name + " połączony");

            return true;
        }

        public void PostNote(string note)
        {
            IChatClient clientCallback =
                OperationContext.Current.GetCallbackChannel<IChatClient>();
            string name = clientsAndNames[clientCallback];

            Console.WriteLine("{0}: {1}", name, note);

            // Metoda ToArray() tworzy kopię kolekcji. Pozwala to uniknąć
            // wyjątku związanego z modyfikacją kolekcji, która nastąpiłaby
            // w wyniku przerwania połączenia z klientem podczas wykonywania pętli.
            KeyValuePair<IChatClient, string>[] copiedNames =
                clientsAndNames.ToArray();
            foreach (KeyValuePair<IChatClient, string> client in copiedNames)
            {
                // Uniknięcie odsyłania komunikatu do klienta, który właśnie go
                // wysłał - nadawca zna treść notatki, którą przed chwilą wprowadził.
                if (client.Key != clientCallback)
                {
                    Console.WriteLine("Wysyłanie notatki do {0}", client.Value);
                    try
                    {
                        client.Key.NotePosted(name, note);
                    }
                    catch (Exception x)
                    {
                        Console.WriteLine("Błąd: {0}", x);
                        DisconnectClient(client.Key);
                    }
                }
            }
        }

        public void Disconnect()
        {
            IChatClient clientCallback =
                OperationContext.Current.GetCallbackChannel<IChatClient>();
            DisconnectClient(clientCallback);
        }

        private void DisconnectClient(IChatClient clientCallback)
        {
            string name = clientsAndNames[clientCallback];
            Console.WriteLine(name + " rozłączony");
            clientsAndNames.Remove(clientCallback);
        }
    }
}
